// Content script that injects the ZENTRADER overlay onto trading pages
let trackerOverlay = null;
let isDragging = false;
let offsetX = 0;
let offsetY = 0;

// Listen for messages from popup
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'toggleTracker') {
    if (trackerOverlay) {
      removeTracker();
      sendResponse({ success: true, isActive: false });
    } else {
      createTracker();
      sendResponse({ success: true, isActive: true });
    }
  } else if (request.action === 'checkTrackerStatus') {
    sendResponse({ isActive: trackerOverlay !== null });
  }
  return true;
});

function createTracker() {
  if (trackerOverlay) return;

  // Create overlay container with vertical layout size
  trackerOverlay = document.createElement('div');
  trackerOverlay.id = 'zentrader-overlay';
  trackerOverlay.style.cssText = `
    position: fixed;
    top: 20px;
    right: 20px;
    width: 450px;
    height: 850px;
    background: rgba(0, 0, 0, 0.98);
    border: 2px solid #6366F1;
    border-radius: 8px;
    z-index: 2147483647;
    box-shadow: 0 0 10px rgba(99, 102, 241, 0.3), 0 0 20px rgba(99, 102, 241, 0.15);
    display: flex;
    flex-direction: column;
    font-family: 'Courier New', monospace;
    overflow: hidden;
    pointer-events: auto;
    resize: both;
    min-width: 400px;
    min-height: 700px;
    max-width: 600px;
    max-height: 1000px;
  `;

  // Create header with drag handle
  const header = document.createElement('div');
  header.id = 'tracker-header';
  header.style.cssText = `
    background: linear-gradient(135deg, #001a0f 0%, #003320 100%);
    padding: 12px;
    cursor: move;
    border-bottom: 2px solid #6366F1;
    display: flex;
    justify-content: space-between;
    align-items: center;
    user-select: none;
    -webkit-user-select: none;
  `;
  header.innerHTML = `
    <span style="color: #6366F1; font-weight: bold; font-size: 13px; text-shadow: 0 0 10px rgba(99, 102, 241,0.5);">📊 ZENTRADER</span>
    <button id="close-tracker" style="
      background: #ff0000;
      color: white;
      border: none;
      padding: 5px 10px;
      cursor: pointer;
      border-radius: 4px;
      font-size: 11px;
      font-weight: bold;
      box-shadow: 0 0 10px rgba(255,0,0,0.3);
    ">✕ CLOSE</button>
  `;

  // Create iframe for tracker content
  const iframe = document.createElement('iframe');
  iframe.setAttribute('allow', 'camera; microphone');
  iframe.style.cssText = `
    width: 100%;
    height: 100%;
    border: none;
    background: #000000;
    pointer-events: auto;
  `;
  iframe.src = chrome.runtime.getURL('floating-compact.html');

  trackerOverlay.appendChild(header);
  trackerOverlay.appendChild(iframe);
  document.body.appendChild(trackerOverlay);

  // Add drag functionality to header only
  header.addEventListener('mousedown', startDrag);
  
  // Add close button functionality
  document.getElementById('close-tracker').addEventListener('click', (e) => {
    e.stopPropagation();
    removeTracker();
  });

  // Store state
  chrome.storage.local.set({ trackerActive: true });
  
  console.log('ZENTRADER overlay created successfully');
}

function removeTracker() {
  if (trackerOverlay) {
    trackerOverlay.remove();
    trackerOverlay = null;
    
    // Remove event listeners
    document.removeEventListener('mousemove', drag);
    document.removeEventListener('mouseup', stopDrag);
    
    chrome.storage.local.set({ trackerActive: false });
    console.log('ZENTRADER overlay removed');
  }
}

function startDrag(e) {
  if (e.target.id === 'close-tracker') return; // Don't drag when clicking close button
  
  isDragging = true;
  
  // Calculate offset from mouse to overlay position
  const rect = trackerOverlay.getBoundingClientRect();
  offsetX = e.clientX - rect.left;
  offsetY = e.clientY - rect.top;
  
  // Add global listeners
  document.addEventListener('mousemove', drag);
  document.addEventListener('mouseup', stopDrag);
  
  // Prevent text selection while dragging
  e.preventDefault();
}

function drag(e) {
  if (!isDragging) return;
  
  e.preventDefault();
  
  // Calculate new position
  let newX = e.clientX - offsetX;
  let newY = e.clientY - offsetY;
  
  // Keep overlay within viewport bounds
  const maxX = window.innerWidth - trackerOverlay.offsetWidth;
  const maxY = window.innerHeight - trackerOverlay.offsetHeight;
  
  newX = Math.max(0, Math.min(newX, maxX));
  newY = Math.max(0, Math.min(newY, maxY));
  
  trackerOverlay.style.left = newX + 'px';
  trackerOverlay.style.top = newY + 'px';
  trackerOverlay.style.right = 'auto';
  trackerOverlay.style.bottom = 'auto';
}

function stopDrag() {
  isDragging = false;
  document.removeEventListener('mousemove', drag);
  document.removeEventListener('mouseup', stopDrag);
}

// Resize functionality - supports all 8 directions
let isResizingOverlay = false;
let resizeDirection = '';
let resizeStartX, resizeStartY, resizeStartWidth, resizeStartHeight, resizeStartLeft, resizeStartTop;

function startResizeOverlay(startX, startY, direction) {
  if (!trackerOverlay) return;
  
  isResizingOverlay = true;
  resizeDirection = direction || 'se';
  resizeStartX = startX;
  resizeStartY = startY;
  
  const rect = trackerOverlay.getBoundingClientRect();
  resizeStartWidth = rect.width;
  resizeStartHeight = rect.height;
  resizeStartLeft = rect.left;
  resizeStartTop = rect.top;
  
  document.addEventListener('mousemove', resizeOverlay);
  document.addEventListener('mouseup', stopResizeOverlay);
  
  // Prevent text selection during resize
  document.body.style.userSelect = 'none';
}

function resizeOverlay(e) {
  if (!isResizingOverlay || !trackerOverlay) return;
  
  const deltaX = e.clientX - resizeStartX;
  const deltaY = e.clientY - resizeStartY;
  
  let newWidth = resizeStartWidth;
  let newHeight = resizeStartHeight;
  let newLeft = resizeStartLeft;
  let newTop = resizeStartTop;
  
  // Handle different resize directions
  if (resizeDirection.includes('e')) {
    newWidth = resizeStartWidth + deltaX;
  }
  if (resizeDirection.includes('w')) {
    newWidth = resizeStartWidth - deltaX;
    newLeft = resizeStartLeft + deltaX;
  }
  if (resizeDirection.includes('s')) {
    newHeight = resizeStartHeight + deltaY;
  }
  if (resizeDirection.includes('n')) {
    newHeight = resizeStartHeight - deltaY;
    newTop = resizeStartTop + deltaY;
  }
  
  // Apply constraints
  newWidth = Math.max(350, Math.min(800, newWidth));
  newHeight = Math.max(500, Math.min(1200, newHeight));
  
  // Apply new dimensions
  trackerOverlay.style.width = newWidth + 'px';
  trackerOverlay.style.height = newHeight + 'px';
  
  // Update position if resizing from top or left
  if (resizeDirection.includes('w') || resizeDirection.includes('n')) {
    if (resizeDirection.includes('w')) {
      trackerOverlay.style.left = newLeft + 'px';
      trackerOverlay.style.right = 'auto';
    }
    if (resizeDirection.includes('n')) {
      trackerOverlay.style.top = newTop + 'px';
      trackerOverlay.style.bottom = 'auto';
    }
  }
}

function stopResizeOverlay() {
  isResizingOverlay = false;
  resizeDirection = '';
  document.removeEventListener('mousemove', resizeOverlay);
  document.removeEventListener('mouseup', stopResizeOverlay);
  document.body.style.userSelect = '';
  
  // Save size preference
  if (trackerOverlay) {
    chrome.storage.local.set({
      overlayWidth: trackerOverlay.style.width,
      overlayHeight: trackerOverlay.style.height
    });
  }
}

// Listen for messages from iframe (floating-compact.html)
window.addEventListener('message', (event) => {
  if (!event.data || !event.data.action) return;
  
  if (event.data.action === 'openFullTracker') {
    // Open fullpage.html in new tab
    const fullTrackerUrl = chrome.runtime.getURL('fullpage.html');
    window.open(fullTrackerUrl, '_blank');
  } else if (event.data.action === 'startResize') {
    // Start resize from iframe with direction
    startResizeOverlay(event.data.x, event.data.y, event.data.direction);
  }
});

// Load saved size on startup
chrome.storage.local.get(['overlayWidth', 'overlayHeight'], (result) => {
  if (trackerOverlay && result.overlayWidth && result.overlayHeight) {
    trackerOverlay.style.width = result.overlayWidth;
    trackerOverlay.style.height = result.overlayHeight;
  }
});

// Check if tracker should be active on page load
chrome.storage.local.get(['trackerActive'], (result) => {
  if (result.trackerActive) {
    // Small delay to ensure page is ready
    setTimeout(createTracker, 500);
  }
});

console.log('ZENTRADER content script loaded');
