// floating-compact.js - Simplified tracker for floating overlay
// COPIED WORKING CODE FROM simple-fake-emotions.js

console.log('=== FLOATING-COMPACT.JS LOADED ===');

let emotions = { stress: 35, excitement: 45, focus: 65, positive: 55 };

const scenarios = [
  { stress: 20, excitement: 25, focus: 75, positive: 55, duration: 8000 },
  { stress: 30, excitement: 20, focus: 85, positive: 50, duration: 10000 },
  { stress: 35, excitement: 65, focus: 60, positive: 70, duration: 6000 },
  { stress: 70, excitement: 55, focus: 40, positive: 30, duration: 5000 },
  { stress: 45, excitement: 40, focus: 55, positive: 45, duration: 7000 }
];

let currentScenario = 0;
let scenarioStartTime = Date.now();

// Setup camera feed - COPIED FROM simple-fake-emotions.js
async function setupCamera() {
  const video = document.getElementById('video');
  const canvas = document.getElementById('canvas');
  
  if (!video || !canvas) {
    console.log('Video/canvas elements not found - skipping camera');
    return;
  }
  
  try {
    const stream = await navigator.mediaDevices.getUserMedia({
      video: { 
        width: 640, 
        height: 480,
        facingMode: 'user'
      }
    });
    
    video.srcObject = stream;
    
    return new Promise((resolve) => {
      video.onloadedmetadata = () => {
        video.play();
        updateCanvasSize();
        console.log('Camera initialized:', video.videoWidth, 'x', video.videoHeight);
        
        // Start fake tracking animation
        drawFakeTracking();
        
        resolve();
      };
    });
  } catch (error) {
    console.error('Camera access error:', error);
  }
}

// Draw fake tracking box on canvas - COPIED FROM simple-fake-emotions.js
function drawFakeTracking() {
  const canvas = document.getElementById('canvas');
  const ctx = canvas?.getContext('2d');
  
  if (!ctx) return;
  
  const time = Date.now() / 1000;
  const centerX = canvas.width / 2;
  const centerY = canvas.height / 2.5;
  const boxWidth = canvas.width * 0.35;
  const boxHeight = canvas.height * 0.45;
  
  // Animate position slightly
  const offsetX = Math.sin(time * 0.5) * 15;
  const offsetY = Math.cos(time * 0.4) * 10;
  
  ctx.clearRect(0, 0, canvas.width, canvas.height);
  
  // Draw green tracking box
  ctx.strokeStyle = '#6366F1';
  ctx.lineWidth = 3;
  ctx.strokeRect(
    centerX - boxWidth/2 + offsetX,
    centerY - boxHeight/2 + offsetY,
    boxWidth,
    boxHeight
  );
  
  // Draw "TRACKING..." label
  ctx.fillStyle = '#6366F1';
  ctx.font = 'bold 12px Courier New';
  ctx.fillText('TRACKING...', centerX - boxWidth/2 + offsetX, centerY - boxHeight/2 + offsetY - 10);
  
  requestAnimationFrame(drawFakeTracking);
}

// Update emotions - COPIED FROM simple-fake-emotions.js
function updateEmotions() {
  const now = Date.now();
  const elapsed = now - scenarioStartTime;
  
  // Switch scenario
  if (elapsed > scenarios[currentScenario].duration) {
    currentScenario = (currentScenario + 1) % scenarios.length;
    scenarioStartTime = now;
    console.log('Scenario changed:', currentScenario);
  }
  
  const target = scenarios[currentScenario];
  
  // Smooth transition
  emotions.stress += (target.stress - emotions.stress) * 0.02;
  emotions.excitement += (target.excitement - emotions.excitement) * 0.02;
  emotions.focus += (target.focus - emotions.focus) * 0.02;
  emotions.positive += (target.positive - emotions.positive) * 0.02;
  
  // Random fluctuation
  emotions.stress += (Math.random() - 0.5) * 2;
  emotions.excitement += (Math.random() - 0.5) * 2;
  emotions.focus += (Math.random() - 0.5) * 1.5;
  emotions.positive += (Math.random() - 0.5) * 1.5;
  
  // Clamp
  emotions.stress = Math.max(10, Math.min(95, emotions.stress));
  emotions.excitement = Math.max(10, Math.min(95, emotions.excitement));
  emotions.focus = Math.max(10, Math.min(95, emotions.focus));
  emotions.positive = Math.max(10, Math.min(95, emotions.positive));
  
  // Update UI directly
  updateBars();
}

// Store previous values for animated counting
let previousValues = { stress: 35, excitement: 45, focus: 65, positive: 55 };

function animateNumber(element, targetValue) {
  if (!element) return;
  
  const currentValue = parseFloat(element.textContent) || 0;
  const diff = targetValue - currentValue;
  
  if (Math.abs(diff) < 0.5) {
    element.textContent = Math.round(targetValue) + '%';
    return;
  }
  
  const step = diff * 0.15;
  const newValue = currentValue + step;
  element.textContent = Math.round(newValue) + '%';
}

function updateBars() {
  // Update stress
  const stressBar = document.getElementById('stress-bar');
  const stressValue = document.getElementById('stress-value');
  const stressRow = document.querySelector('.emotion-row.stress-row');
  
  if (stressBar && stressValue) {
    stressBar.style.width = Math.round(emotions.stress) + '%';
    animateNumber(stressValue, emotions.stress);
    
    // High stress visual feedback
    if (stressRow) {
      if (emotions.stress >= 80) {
        stressRow.classList.add('critical-stress');
        stressRow.classList.remove('high-stress');
      } else if (emotions.stress >= 70) {
        stressRow.classList.add('high-stress');
        stressRow.classList.remove('critical-stress');
      } else {
        stressRow.classList.remove('high-stress', 'critical-stress');
      }
    }
  }
  
  // Update excitement
  const excitementBar = document.getElementById('excitement-bar');
  const excitementValue = document.getElementById('excitement-value');
  if (excitementBar && excitementValue) {
    excitementBar.style.width = Math.round(emotions.excitement) + '%';
    animateNumber(excitementValue, emotions.excitement);
  }
  
  // Update focus
  const focusBar = document.getElementById('focus-bar');
  const focusValue = document.getElementById('focus-value');
  if (focusBar && focusValue) {
    focusBar.style.width = Math.round(emotions.focus) + '%';
    animateNumber(focusValue, emotions.focus);
  }
  
  // Update positive
  const positiveBar = document.getElementById('positive-bar');
  const positiveValue = document.getElementById('positive-value');
  if (positiveBar && positiveValue) {
    positiveBar.style.width = Math.round(emotions.positive) + '%';
    animateNumber(positiveValue, emotions.positive);
  }
  
  // Store current values
  previousValues = {
    stress: emotions.stress,
    excitement: emotions.excitement,
    focus: emotions.focus,
    positive: emotions.positive
  };
}

// Update canvas size to match video
function updateCanvasSize() {
  const video = document.getElementById('video');
  const canvas = document.getElementById('canvas');
  
  if (video && canvas && video.videoWidth > 0) {
    canvas.width = video.videoWidth;
    canvas.height = video.videoHeight;
    canvas.style.width = video.offsetWidth + 'px';
    canvas.style.height = video.offsetHeight + 'px';
  }
}

// Setup control buttons and resize handles
function setupControls() {
  // Open Full Tracker button
  const openFullBtn = document.getElementById('open-full-btn');
  if (openFullBtn) {
    openFullBtn.addEventListener('click', () => {
      console.log('Open Full Tracker clicked');
      // Send message to parent window (content script) to open full tracker
      window.parent.postMessage({ action: 'openFullTracker' }, '*');
    });
  }
  
  // Setup all resize handles (8 directions)
  const resizeHandles = document.querySelectorAll('.resize-handle');
  resizeHandles.forEach(handle => {
    handle.addEventListener('mousedown', (e) => {
      e.preventDefault();
      e.stopPropagation();
      const direction = handle.getAttribute('data-direction');
      // Send message to parent to start resize with direction
      window.parent.postMessage({ 
        action: 'startResize', 
        direction: direction,
        x: e.clientX, 
        y: e.clientY 
      }, '*');
    });
  });
  
  // Update canvas size on window resize
  window.addEventListener('resize', updateCanvasSize);
}

// Initialize everything - COPIED FROM simple-fake-emotions.js
async function init() {
  console.log('Starting floating compact tracker...');
  
  // Setup control buttons
  setupControls();
  
  // Setup camera
  await setupCamera();
  
  // Force first update
  setTimeout(() => {
    updateBars();
    console.log('First update done');
  }, 500);
  
  // Update every 100ms
  setInterval(updateEmotions, 100);
  
  console.log('Floating compact tracker running');
}

// Start when DOM is ready
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', init);
} else {
  init();
}
