// fullpage.js - Full tracker interface controller

let chart;
let pageSessionStartTime = Date.now();
let updateInterval;
let alertCount = 0;

// Statistics tracking
let stats = {
  avgStress: 0,
  avgExcitement: 0,
  avgFocus: 0,
  peakStress: 0,
  peakExcitement: 0,
  timeOptimal: 0,
  totalTime: 0
};

let zoneTimeTracking = {
  optimal: 0,
  caution: 0,
  danger: 0
};

let currentZone = 'optimal';
let lastZoneChange = Date.now();

// Initialize
document.addEventListener('DOMContentLoaded', async () => {
  console.log('Initializing full tracker...');
  
  // Setup Chart.js
  setupChart();
  
  // Setup Axiom button
  document.getElementById('axiom-btn').addEventListener('click', () => {
    window.open('https://axiom.trade', '_blank');
  });
  
  // Simple fake emotions - no complex tracker needed
  console.log('Using simple fake emotions (no real tracking)');
  addAlert('Demo mode: Showing simulated emotions', 'info');
  
  // Update session time every second
  setInterval(updateSessionTime, 1000);
  
  // Update statistics every 5 seconds
  setInterval(updateStatistics, 5000);
  
  // Note: simple-fake-emotions.js handles the emotion bar updates directly
  // No need for updateUI loop here
});

// Setup Chart.js emotion history graph
function setupChart() {
  const ctx = document.getElementById('emotionChart');
  
  chart = new Chart(ctx, {
    type: 'line',
    data: {
      labels: Array(60).fill(''),
      datasets: [
        {
          label: 'Stress',
          data: Array(60).fill(0),
          borderColor: '#ff4444',
          backgroundColor: 'rgba(255, 68, 68, 0.1)',
          tension: 0.4,
          borderWidth: 2
        },
        {
          label: 'Excitement',
          data: Array(60).fill(0),
          borderColor: '#ffaa00',
          backgroundColor: 'rgba(255, 170, 0, 0.1)',
          tension: 0.4,
          borderWidth: 2
        },
        {
          label: 'Focus',
          data: Array(60).fill(0),
          borderColor: '#3B82F6',
          backgroundColor: 'rgba(0, 187, 255, 0.1)',
          tension: 0.4,
          borderWidth: 2
        },
        {
          label: 'Positive',
          data: Array(60).fill(0),
          borderColor: '#6366F1',
          backgroundColor: 'rgba(99, 102, 241, 0.1)',
          tension: 0.4,
          borderWidth: 2
        }
      ]
    },
    options: {
      responsive: true,
      maintainAspectRatio: false,
      plugins: {
        legend: {
          display: true,
          position: 'top',
          labels: {
            color: '#6366F1',
            font: {
              family: 'Courier New',
              size: 11
            }
          }
        }
      },
      scales: {
        y: {
          beginAtZero: true,
          max: 100,
          ticks: {
            color: '#6366F1',
            font: {
              family: 'Courier New',
              size: 10
            }
          },
          grid: {
            color: 'rgba(99, 102, 241, 0.1)'
          }
        },
        x: {
          ticks: {
            color: '#6366F1',
            font: {
              family: 'Courier New',
              size: 10
            }
          },
          grid: {
            color: 'rgba(99, 102, 241, 0.1)'
          }
        }
      }
    }
  });
}

// Update UI with current emotions
let lastLogTime = 0;
function updateUI() {
  if (!window.EmotionTracker) {
    console.error('EmotionTracker API not available');
    return;
  }
  
  const emotions = window.EmotionTracker.getEmotions();
  
  if (!emotions) {
    console.error('No emotions returned from tracker');
    return;
  }
  
  // Log only once per second to avoid spam
  const now = Date.now();
  if (now - lastLogTime > 1000) {
    console.log('updateUI called with emotions:', emotions);
    lastLogTime = now;
  }
  
  // Update emotion bars
  updateBar('stress', emotions.stress);
  updateBar('excitement', emotions.excitement);
  updateBar('focus', emotions.focus);
  updateBar('positive', emotions.positive);
  
  // Update chart
  updateChart();
  
  // Update trading zone
  updateTradingZone(emotions);
  
  // Update emotional state
  updateEmotionalState(emotions);
  
  // Update guidance
  updateGuidance(emotions);
}

// Update emotion bar
function updateBar(emotion, value) {
  const bar = document.getElementById(`${emotion}-bar`);
  const valueLabel = document.getElementById(`${emotion}-value`);
  
  if (bar && valueLabel) {
    bar.style.width = `${value}%`;
    valueLabel.textContent = `${Math.round(value)}%`;
  } else {
    console.error(`Missing elements for ${emotion}: bar=${!!bar}, valueLabel=${!!valueLabel}`);
  }
}

// Update chart with emotion history
function updateChart() {
  const history = window.EmotionTracker.getEmotionHistory();
  
  chart.data.datasets[0].data = history.stress;
  chart.data.datasets[1].data = history.excitement;
  chart.data.datasets[2].data = history.focus;
  chart.data.datasets[3].data = history.positive;
  
  chart.update('none'); // Update without animation for performance
}

// Update trading zone indicator
function updateTradingZone(emotions) {
  const zoneIndicator = document.getElementById('zone-indicator');
  const zoneIcon = zoneIndicator.querySelector('.zone-icon');
  const zoneText = zoneIndicator.querySelector('.zone-text');
  const { stress, excitement, focus } = emotions;
  
  let newZone;
  
  // Determine zone
  if (stress > 70 || excitement > 70 || focus < 25) {
    newZone = 'danger';
    zoneIndicator.className = 'zone-card danger';
    zoneIcon.textContent = '🚫';
    zoneText.textContent = 'DANGER ZONE - STOP TRADING';
  } else if (stress > 45 || excitement > 45 || focus < 40) {
    newZone = 'caution';
    zoneIndicator.className = 'zone-card caution';
    zoneIcon.textContent = '⚠️';
    zoneText.textContent = 'CAUTION ZONE - REDUCE POSITION SIZE';
  } else {
    newZone = 'optimal';
    zoneIndicator.className = 'zone-card optimal';
    zoneIcon.textContent = '✅';
    zoneText.textContent = 'OPTIMAL TRADING ZONE';
  }
  
  // Track zone changes
  if (newZone !== currentZone) {
    const timeInZone = Date.now() - lastZoneChange;
    zoneTimeTracking[currentZone] += timeInZone;
    
    currentZone = newZone;
    lastZoneChange = Date.now();
    
    // Add alert
    if (newZone === 'danger') {
      addAlert('Entered DANGER zone - Consider stopping trading', 'danger');
    } else if (newZone === 'caution') {
      addAlert('Entered CAUTION zone - Reduce risk', 'warning');
    } else {
      addAlert('Entered OPTIMAL zone', 'info');
    }
  }
}

// Update emotional state
function updateEmotionalState(emotions) {
  const stateLabel = document.getElementById('emotional-state');
  const { stress, focus } = emotions;
  
  if (stress > 60 && focus > 50) {
    stateLabel.textContent = '🎯 FOCUSED';
    stateLabel.style.color = 'var(--accent-green)';
  } else if (stress > 60) {
    stateLabel.textContent = '⚠️ STRESSED';
    stateLabel.style.color = 'var(--danger)';
  } else {
    stateLabel.textContent = '✓ NEUTRAL';
    stateLabel.style.color = 'var(--text-secondary)';
  }
}

// Update guidance message
function updateGuidance(emotions) {
  const guidanceLabel = document.getElementById('guidance');
  const { stress, excitement, focus } = emotions;
  
  let message = '';
  
  if (stress > 70) {
    message = '💡 GUIDANCE: High stress detected. Take a break, practice deep breathing, and step away from trading.';
  } else if (excitement > 70) {
    message = '💡 GUIDANCE: High excitement detected. Avoid impulsive decisions. Stick to your trading plan.';
  } else if (focus < 30) {
    message = '💡 GUIDANCE: Low focus detected. Take a short break, hydrate, or do a quick exercise.';
  } else if (stress > 45 || excitement > 45) {
    message = '💡 GUIDANCE: Moderate emotional state. Reduce position sizes and trade conservatively.';
  } else {
    message = '💡 GUIDANCE: Maintain your current state for optimal trading performance. Stay disciplined.';
  }
  
  guidanceLabel.textContent = message;
}

// Update session time
function updateSessionTime() {
  const elapsed = Date.now() - sessionStartTime;
  const minutes = Math.floor(elapsed / 60000);
  const seconds = Math.floor((elapsed % 60000) / 1000);
  
  const timeLabel = document.getElementById('session-time');
  timeLabel.textContent = `${String(minutes).padStart(2, '0')}:${String(seconds).padStart(2, '0')}`;
}

// Update statistics
function updateStatistics() {
  const history = window.EmotionTracker.getEmotionHistory();
  
  if (history.stress.length === 0) return;
  
  // Calculate averages
  stats.avgStress = average(history.stress);
  stats.avgExcitement = average(history.excitement);
  stats.avgFocus = average(history.focus);
  
  // Calculate peaks
  stats.peakStress = Math.max(...history.stress);
  stats.peakExcitement = Math.max(...history.excitement);
  
  // Calculate time in optimal zone
  const totalTime = Date.now() - sessionStartTime;
  const optimalPercent = (zoneTimeTracking.optimal / totalTime) * 100;
  
  // Update UI
  document.getElementById('avg-stress').textContent = `${Math.round(stats.avgStress)}%`;
  document.getElementById('avg-excitement').textContent = `${Math.round(stats.avgExcitement)}%`;
  document.getElementById('avg-focus').textContent = `${Math.round(stats.avgFocus)}%`;
  document.getElementById('peak-stress').textContent = `${Math.round(stats.peakStress)}%`;
  document.getElementById('peak-excitement').textContent = `${Math.round(stats.peakExcitement)}%`;
  document.getElementById('time-optimal').textContent = `${Math.round(optimalPercent)}%`;
}

// Add alert to history
function addAlert(message, severity = 'info') {
  const alertList = document.getElementById('alert-list');
  const timestamp = new Date().toLocaleTimeString();
  
  const alertItem = document.createElement('div');
  alertItem.className = `alert-item ${severity}`;
  alertItem.textContent = `[${timestamp}] ${message}`;
  
  // Add to top
  alertList.insertBefore(alertItem, alertList.firstChild);
  
  // Keep only last 20 alerts
  while (alertList.children.length > 20) {
    alertList.removeChild(alertList.lastChild);
  }
  
  alertCount++;
}

// Helper: Calculate average
function average(arr) {
  if (arr.length === 0) return 0;
  return arr.reduce((a, b) => a + b, 0) / arr.length;
}

// Cleanup on page unload
window.addEventListener('beforeunload', () => {
  if (updateInterval) {
    clearInterval(updateInterval);
  }
  window.EmotionTracker.stop();
});
