// Popup script for toggling the tracker overlay
const toggleBtn = document.getElementById('toggleBtn');
const statusDiv = document.getElementById('status');
const instructionDiv = document.getElementById('instruction');

// Check current tracker status
async function updateStatus() {
  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    
    if (!tab || !tab.id) {
      statusDiv.textContent = 'Tracker: No active tab';
      statusDiv.classList.remove('active');
      toggleBtn.textContent = 'START TRACKER OVERLAY';
      toggleBtn.classList.remove('stop');
      instructionDiv.textContent = 'Open a webpage to use tracker';
      return;
    }

    // Check if this is a restricted page
    if (tab.url && (tab.url.startsWith('chrome://') || tab.url.startsWith('chrome-extension://') || tab.url.startsWith('edge://') || tab.url.startsWith('about:'))) {
      statusDiv.textContent = 'Tracker: Cannot run here';
      statusDiv.classList.remove('active');
      toggleBtn.textContent = 'START TRACKER OVERLAY';
      toggleBtn.classList.remove('stop');
      toggleBtn.disabled = true;
      toggleBtn.style.opacity = '0.5';
      instructionDiv.textContent = 'Extensions cannot run on browser pages. Please open a regular website (e.g., google.com, axiom.trade)';
      return;
    } else {
      toggleBtn.disabled = false;
      toggleBtn.style.opacity = '1';
    }

    try {
      const response = await chrome.tabs.sendMessage(tab.id, { action: 'checkTrackerStatus' });
      
      if (response && response.isActive) {
        toggleBtn.textContent = '⏹ STOP TRACKER OVERLAY';
        toggleBtn.classList.add('stop');
        statusDiv.textContent = 'Tracker: Active ✓';
        statusDiv.classList.add('active');
        instructionDiv.textContent = 'Tracker is visible on your page';
      } else {
        toggleBtn.textContent = '▶ START TRACKER OVERLAY';
        toggleBtn.classList.remove('stop');
        statusDiv.textContent = 'Tracker: Inactive';
        statusDiv.classList.remove('active');
        instructionDiv.textContent = 'Click START to activate overlay';
      }
    } catch (error) {
      // Content script not loaded yet or page doesn't support it
      toggleBtn.textContent = '▶ START TRACKER OVERLAY';
      toggleBtn.classList.remove('stop');
      statusDiv.textContent = 'Tracker: Ready';
      statusDiv.classList.remove('active');
      instructionDiv.textContent = 'Click START to activate overlay';
    }
  } catch (error) {
    console.error('Error updating status:', error);
    statusDiv.textContent = 'Tracker: Error';
    instructionDiv.textContent = 'Please refresh the page';
  }
}

// Toggle tracker
toggleBtn.addEventListener('click', async () => {
  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    
    if (!tab || !tab.id) {
      alert('Please open a webpage first');
      return;
    }

    // Disable button during operation
    toggleBtn.disabled = true;
    toggleBtn.style.opacity = '0.6';

    try {
      const response = await chrome.tabs.sendMessage(tab.id, { action: 'toggleTracker' });
      
      if (response && response.success) {
        // Update UI based on new state
        setTimeout(updateStatus, 200);
      } else {
        throw new Error('Failed to toggle tracker');
      }
    } catch (error) {
      console.error('Error toggling tracker:', error);
      // Check if it's an extension context invalidation error
      if (error.message && error.message.includes('Extension context invalidated')) {
        alert('Extension was reloaded. Please close this popup and try again.');
      } else if (error.message && error.message.includes('Could not establish connection')) {
        alert('Please refresh the page and try again.');
      } else {
        alert('Error: Please refresh the page and try again.\n\nIf the problem persists, the page may not support extensions.');
      }
    } finally {
      // Re-enable button
      toggleBtn.disabled = false;
      toggleBtn.style.opacity = '1';
    }
  } catch (error) {
    console.error('Error in toggle handler:', error);
    toggleBtn.disabled = false;
    toggleBtn.style.opacity = '1';
  }
});

// Update status when popup opens
updateStatus();

// Update status every 500ms while popup is open
setInterval(updateStatus, 500);
