// Demo Tracker - Convincing fake face tracking for marketing/demos
// Animated tracking box + scenario-based emotions

console.log('=== DEMO TRACKER LOADED ===');

// Wait for DOM to be ready
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', initTracker);
} else {
  initTracker();
}

function initTracker() {
  console.log('=== INITIALIZING TRACKER - DOM READY ===');

let video, canvas, ctx;
let isTracking = false;

// Fake tracking box position
let fakeBoxPos = { x: 0, y: 0, width: 0, height: 0 };
let targetBoxPos = { x: 0, y: 0 };

// Emotions - start with visible values
let emotions = {
  stress: 35,
  excitement: 45,
  focus: 65,
  positive: 55
};

// Emotion scenarios that cycle through
const scenarios = [
  { name: 'calm', stress: 20, excitement: 25, focus: 75, positive: 55, duration: 8000 },
  { name: 'focused', stress: 30, excitement: 20, focus: 85, positive: 50, duration: 10000 },
  { name: 'excited', stress: 35, excitement: 65, focus: 60, positive: 70, duration: 6000 },
  { name: 'stressed', stress: 70, excitement: 55, focus: 40, positive: 30, duration: 5000 },
  { name: 'recovering', stress: 45, excitement: 40, focus: 55, positive: 45, duration: 7000 }
];

let currentScenario = 0;
let scenarioStartTime = Date.now();

// Session tracking
let sessionStartTime = Date.now();
let emotionHistory = {
  stress: [],
  excitement: [],
  focus: [],
  positive: []
};
const MAX_HISTORY = 60;

// Setup camera
async function setupCamera() {
  console.log('=== SETUP DEMO CAMERA ===');
  video = document.getElementById('video');
  canvas = document.getElementById('canvas');
  
  if (!video || !canvas) {
    console.error('Video or canvas element not found');
    return false;
  }
  
  ctx = canvas.getContext('2d');
  
  try {
    const stream = await navigator.mediaDevices.getUserMedia({
      video: { width: 640, height: 480, facingMode: 'user' }
    });
    
    video.srcObject = stream;
    
    return new Promise((resolve) => {
      video.onloadedmetadata = () => {
        video.play();
        canvas.width = video.videoWidth;
        canvas.height = video.videoHeight;
        
        // Initialize fake box in center
        fakeBoxPos.width = canvas.width * 0.35;
        fakeBoxPos.height = canvas.height * 0.45;
        fakeBoxPos.x = (canvas.width - fakeBoxPos.width) / 2;
        fakeBoxPos.y = (canvas.height - fakeBoxPos.height) / 2.5;
        
        targetBoxPos.x = fakeBoxPos.x;
        targetBoxPos.y = fakeBoxPos.y;
        
        console.log('Demo camera ready:', canvas.width, 'x', canvas.height);
        resolve(true);
      };
    });
  } catch (error) {
    console.error('Camera error:', error);
    return false;
  }
}

// Animate fake tracking box
function animateFakeTracking() {
  if (!isTracking || !ctx) {
    requestAnimationFrame(animateFakeTracking);
    return;
  }
  
  // Slowly move target position (simulate natural head movement)
  if (Math.random() < 0.02) {
    const moveRange = 30;
    targetBoxPos.x = fakeBoxPos.x + (Math.random() - 0.5) * moveRange;
    targetBoxPos.y = fakeBoxPos.y + (Math.random() - 0.5) * moveRange;
    
    // Keep within bounds
    targetBoxPos.x = Math.max(20, Math.min(canvas.width - fakeBoxPos.width - 20, targetBoxPos.x));
    targetBoxPos.y = Math.max(20, Math.min(canvas.height - fakeBoxPos.height - 20, targetBoxPos.y));
  }
  
  // Smoothly move current position toward target
  fakeBoxPos.x += (targetBoxPos.x - fakeBoxPos.x) * 0.1;
  fakeBoxPos.y += (targetBoxPos.y - fakeBoxPos.y) * 0.1;
  
  // Slight size variation (simulate distance changes)
  if (Math.random() < 0.01) {
    const sizeVariation = (Math.random() - 0.5) * 0.02;
    const newWidth = fakeBoxPos.width * (1 + sizeVariation);
    const newHeight = fakeBoxPos.height * (1 + sizeVariation);
    
    if (newWidth > canvas.width * 0.25 && newWidth < canvas.width * 0.45) {
      fakeBoxPos.width = newWidth;
      fakeBoxPos.height = newHeight;
    }
  }
  
  // Clear canvas
  ctx.clearRect(0, 0, canvas.width, canvas.height);
  
  // Draw main face box (green)
  ctx.strokeStyle = '#6366F1';
  ctx.lineWidth = 3;
  ctx.strokeRect(
    fakeBoxPos.x,
    fakeBoxPos.y,
    fakeBoxPos.width,
    fakeBoxPos.height
  );
  
  // Draw corner brackets (cyberpunk style)
  const bracketSize = 20;
  ctx.lineWidth = 4;
  
  // Top-left
  ctx.beginPath();
  ctx.moveTo(fakeBoxPos.x, fakeBoxPos.y + bracketSize);
  ctx.lineTo(fakeBoxPos.x, fakeBoxPos.y);
  ctx.lineTo(fakeBoxPos.x + bracketSize, fakeBoxPos.y);
  ctx.stroke();
  
  // Top-right
  ctx.beginPath();
  ctx.moveTo(fakeBoxPos.x + fakeBoxPos.width - bracketSize, fakeBoxPos.y);
  ctx.lineTo(fakeBoxPos.x + fakeBoxPos.width, fakeBoxPos.y);
  ctx.lineTo(fakeBoxPos.x + fakeBoxPos.width, fakeBoxPos.y + bracketSize);
  ctx.stroke();
  
  // Bottom-left
  ctx.beginPath();
  ctx.moveTo(fakeBoxPos.x, fakeBoxPos.y + fakeBoxPos.height - bracketSize);
  ctx.lineTo(fakeBoxPos.x, fakeBoxPos.y + fakeBoxPos.height);
  ctx.lineTo(fakeBoxPos.x + bracketSize, fakeBoxPos.y + fakeBoxPos.height);
  ctx.stroke();
  
  // Bottom-right
  ctx.beginPath();
  ctx.moveTo(fakeBoxPos.x + fakeBoxPos.width - bracketSize, fakeBoxPos.y + fakeBoxPos.height);
  ctx.lineTo(fakeBoxPos.x + fakeBoxPos.width, fakeBoxPos.y + fakeBoxPos.height);
  ctx.lineTo(fakeBoxPos.x + fakeBoxPos.width, fakeBoxPos.y + fakeBoxPos.height - bracketSize);
  ctx.stroke();
  
  // Draw fake facial landmarks
  drawFakeLandmarks();
  
  // Draw "TRACKING" label
  ctx.fillStyle = '#6366F1';
  ctx.font = 'bold 12px Courier New';
  ctx.fillText('TRACKING...', fakeBoxPos.x, fakeBoxPos.y - 10);
  
  requestAnimationFrame(animateFakeTracking);
}

// Draw fake facial landmarks
function drawFakeLandmarks() {
  ctx.fillStyle = '#3B82F6';
  
  const centerX = fakeBoxPos.x + fakeBoxPos.width / 2;
  const centerY = fakeBoxPos.y + fakeBoxPos.height / 2;
  
  // Eyes (cyan dots)
  const eyeY = fakeBoxPos.y + fakeBoxPos.height * 0.35;
  const eyeSpacing = fakeBoxPos.width * 0.25;
  
  // Left eye
  ctx.beginPath();
  ctx.arc(centerX - eyeSpacing, eyeY, 4, 0, 2 * Math.PI);
  ctx.fill();
  
  // Right eye
  ctx.beginPath();
  ctx.arc(centerX + eyeSpacing, eyeY, 4, 0, 2 * Math.PI);
  ctx.fill();
  
  // Nose (yellow dot)
  ctx.fillStyle = '#ffaa00';
  ctx.beginPath();
  ctx.arc(centerX, centerY, 3, 0, 2 * Math.PI);
  ctx.fill();
  
  // Mouth (yellow line)
  const mouthY = fakeBoxPos.y + fakeBoxPos.height * 0.7;
  const mouthWidth = fakeBoxPos.width * 0.2;
  ctx.fillRect(centerX - mouthWidth / 2, mouthY, mouthWidth, 3);
}

// Update emotions based on scenarios
function updateScenarioEmotions() {
  const now = Date.now();
  const elapsed = now - scenarioStartTime;
  
  // Check if should move to next scenario
  if (elapsed > scenarios[currentScenario].duration) {
    currentScenario = (currentScenario + 1) % scenarios.length;
    scenarioStartTime = now;
    console.log('Scenario changed to:', scenarios[currentScenario].name);
  }
  
  const target = scenarios[currentScenario];
  
  // Log every 2 seconds
  if (Math.random() < 0.02) {
    console.log('Scenario:', target.name, 'Current emotions:', {
      stress: Math.round(emotions.stress),
      excitement: Math.round(emotions.excitement),
      focus: Math.round(emotions.focus),
      positive: Math.round(emotions.positive)
    });
  }
  
  // Smoothly transition to target emotions
  const smoothing = 0.02;
  
  emotions.stress += (target.stress - emotions.stress) * smoothing;
  emotions.excitement += (target.excitement - emotions.excitement) * smoothing;
  emotions.focus += (target.focus - emotions.focus) * smoothing;
  emotions.positive += (target.positive - emotions.positive) * smoothing;
  
  // Add small random variations
  emotions.stress += (Math.random() - 0.5) * 2;
  emotions.excitement += (Math.random() - 0.5) * 2;
  emotions.focus += (Math.random() - 0.5) * 1.5;
  emotions.positive += (Math.random() - 0.5) * 1.5;
  
  // Clamp values
  emotions.stress = Math.max(10, Math.min(95, emotions.stress));
  emotions.excitement = Math.max(10, Math.min(95, emotions.excitement));
  emotions.focus = Math.max(10, Math.min(95, emotions.focus));
  emotions.positive = Math.max(10, Math.min(95, emotions.positive));
  
  // Update emotion history
  updateEmotionHistory();
  
  // Save to storage for popup
  chrome.storage.local.set({ emotions, sessionStart: sessionStartTime });
}

// Update emotion history
function updateEmotionHistory() {
  emotionHistory.stress.push(emotions.stress);
  emotionHistory.excitement.push(emotions.excitement);
  emotionHistory.focus.push(emotions.focus);
  emotionHistory.positive.push(emotions.positive);
  
  if (emotionHistory.stress.length > MAX_HISTORY) {
    emotionHistory.stress.shift();
    emotionHistory.excitement.shift();
    emotionHistory.focus.shift();
    emotionHistory.positive.shift();
  }
}

// Get current emotions
function getEmotions() {
  return { ...emotions };
}

// Get emotion history
function getEmotionHistory() {
  return { ...emotionHistory };
}

// Start tracking
async function startTracking() {
  if (isTracking) return true;
  
  console.log('=== STARTING DEMO TRACKER ===');
  
  const cameraReady = await setupCamera();
  if (!cameraReady) {
    console.error('Failed to setup camera');
    return false;
  }
  
  isTracking = true;
  sessionStartTime = Date.now();
  
  // Start animation loops
  animateFakeTracking();
  setInterval(updateScenarioEmotions, 100);
  
  console.log('=== DEMO TRACKER STARTED ===');
  return true;
}

// Stop tracking
function stopTracking() {
  isTracking = false;
  if (video && video.srcObject) {
    const tracks = video.srcObject.getTracks();
    tracks.forEach(track => track.stop());
  }
}

// Export functions
window.EmotionTracker = {
  start: startTracking,
  stop: stopTracking,
  getEmotions,
  getEmotionHistory,
  isTracking: () => isTracking
};

console.log('Demo tracker ready');

// Verify DOM elements exist
setTimeout(() => {
  const stressBar = document.getElementById('stress-bar');
  console.log('Stress bar found:', stressBar);
  console.log('All emotion bars:', document.querySelectorAll('.emotion-bar-fill').length);
}, 100);

} // End of initTracker function
