// tracker.js - Face detection and emotion analysis using TensorFlow.js

console.log('=== TRACKER.JS LOADED ===');
console.log('TensorFlow.js available:', typeof tf !== 'undefined');
console.log('BlazeFace available:', typeof blazeface !== 'undefined');

let model;
let videoElement;
let canvasElement;
let ctx;
let isTracking = false;

// Emotion state
let emotions = {
  stress: 0,
  excitement: 0,
  focus: 0,
  positive: 0
};

// History for smoothing and analysis
let faceHistory = [];
let movementHistory = [];
let faceSizeHistory = [];
const HISTORY_LENGTH = 30; // ~1 second at 30fps

// Previous face position for movement calculation
let prevFacePosition = null;

// Session tracking
let sessionStartTime = Date.now();
let emotionHistory = {
  stress: [],
  excitement: [],
  focus: [],
  positive: []
};
const MAX_HISTORY = 60; // 60 seconds

// Initialize camera and face detection
async function setupCamera() {
  console.log('=== SETUP CAMERA ===');
  videoElement = document.getElementById('video');
  canvasElement = document.getElementById('canvas');
  
  console.log('Video element:', videoElement);
  console.log('Canvas element:', canvasElement);
  
  if (!videoElement) {
    console.error('Video element not found');
    return null;
  }

  try {
    const stream = await navigator.mediaDevices.getUserMedia({
      video: { 
        width: 640, 
        height: 480,
        facingMode: 'user'
      }
    });
    
    videoElement.srcObject = stream;
    console.log('Stream set to video element');
    
    return new Promise((resolve) => {
      videoElement.onloadedmetadata = () => {
        console.log('Video metadata loaded');
        videoElement.play();
        
        // Setup canvas
        if (canvasElement) {
          canvasElement.width = videoElement.videoWidth;
          canvasElement.height = videoElement.videoHeight;
          ctx = canvasElement.getContext('2d');
          console.log('Canvas setup:', canvasElement.width, 'x', canvasElement.height);
        }
        
        console.log('Camera setup complete');
        resolve(videoElement);
      };
    });
  } catch (error) {
    console.error('Camera access denied:', error);
    alert('Please allow camera access to use the ZENTRADER.');
    return null;
  }
}

// Load BlazeFace model
async function loadModel() {
  try {
    console.log('=== LOADING BLAZEFACE MODEL ===');
    console.log('blazeface object type:', typeof blazeface);
    console.log('blazeface.load type:', typeof blazeface?.load);
    
    if (typeof blazeface === 'undefined') {
      console.error('ERROR: blazeface is undefined!');
      return false;
    }
    
    console.log('Calling blazeface.load()...');
    model = await blazeface.load();
    console.log('BlazeFace model loaded successfully:', model);
    return true;
  } catch (error) {
    console.error('Failed to load BlazeFace model:', error);
    console.error('Error stack:', error.stack);
    return false;
  }
}

// Main detection loop
async function detectFace() {
  if (!isTracking || !model || !videoElement) {
    if (!isTracking) console.log('Not tracking yet');
    if (!model) console.log('Model not loaded');
    if (!videoElement) console.log('Video element missing');
    requestAnimationFrame(detectFace);
    return;
  }

  try {
    const predictions = await model.estimateFaces(videoElement, false);
    
    if (predictions.length > 0) {
      console.log('Face detected! Predictions:', predictions.length);
    }
    
    if (predictions.length > 0) {
      const face = predictions[0];
      console.log('Face data:', face);
      analyzeFace(face);
      
      // Draw face box (optional)
      if (ctx && canvasElement) {
        ctx.clearRect(0, 0, canvasElement.width, canvasElement.height);
        drawFaceBox(face);
      }
    } else {
      // No face detected - reduce focus
      emotions.focus = Math.max(0, emotions.focus - 2);
    }
    
    // Update emotion history
    updateEmotionHistory();
    
    // Save to storage for popup
    chrome.storage.local.set({ emotions, sessionStart: sessionStartTime });
    
  } catch (error) {
    console.error('Face detection error:', error);
  }
  
  // Continue loop
  requestAnimationFrame(detectFace);
}

// Analyze face and calculate emotions
function analyzeFace(face) {
  // Extract face metrics
  const faceBox = face.topLeft.concat(face.bottomRight);
  const faceWidth = faceBox[2] - faceBox[0];
  const faceHeight = faceBox[3] - faceBox[1];
  const faceSize = faceWidth * faceHeight;
  const faceCenterX = (faceBox[0] + faceBox[2]) / 2;
  const faceCenterY = (faceBox[1] + faceBox[3]) / 2;
  
  // Store in history
  faceHistory.push({ x: faceCenterX, y: faceCenterY, size: faceSize });
  if (faceHistory.length > HISTORY_LENGTH) {
    faceHistory.shift();
  }
  
  faceSizeHistory.push(faceSize);
  if (faceSizeHistory.length > HISTORY_LENGTH) {
    faceSizeHistory.shift();
  }
  
  // Calculate movement
  let movement = 0;
  if (prevFacePosition) {
    const dx = faceCenterX - prevFacePosition.x;
    const dy = faceCenterY - prevFacePosition.y;
    movement = Math.sqrt(dx * dx + dy * dy);
  }
  prevFacePosition = { x: faceCenterX, y: faceCenterY };
  
  movementHistory.push(movement);
  if (movementHistory.length > HISTORY_LENGTH) {
    movementHistory.shift();
  }
  
  // Calculate emotions
  calculateEmotions(faceSize, movement);
}

// Calculate emotion levels
function calculateEmotions(faceSize, movement) {
  if (faceHistory.length < 10) return;
  
  // Calculate average face size (baseline)
  const avgFaceSize = faceSizeHistory.reduce((a, b) => a + b, 0) / faceSizeHistory.length;
  
  // Calculate average movement
  const avgMovement = movementHistory.reduce((a, b) => a + b, 0) / movementHistory.length;
  
  // STRESS: Based on movement (jittery = stressed)
  const movementStress = Math.min(100, (avgMovement / 5) * 100);
  emotions.stress = smooth(emotions.stress, movementStress, 0.15);
  
  // EXCITEMENT: Based on face size (leaning in = excited)
  const sizeRatio = faceSize / avgFaceSize;
  const sizeExcitement = Math.min(100, Math.max(0, (sizeRatio - 0.9) * 200));
  emotions.excitement = smooth(emotions.excitement, sizeExcitement, 0.1);
  
  // FOCUS: Inverse of movement (still = focused)
  const stabilityFocus = Math.max(0, 100 - movementStress);
  emotions.focus = smooth(emotions.focus, stabilityFocus, 0.1);
  
  // POSITIVE: Based on overall calmness and stability
  const calmness = (stabilityFocus + (100 - movementStress)) / 2;
  emotions.positive = smooth(emotions.positive, calmness * 0.8, 0.08);
  
  // Clamp values
  emotions.stress = Math.max(0, Math.min(100, emotions.stress));
  emotions.excitement = Math.max(0, Math.min(100, emotions.excitement));
  emotions.focus = Math.max(0, Math.min(100, emotions.focus));
  emotions.positive = Math.max(0, Math.min(100, emotions.positive));
}

// Smooth value changes
function smooth(current, target, factor) {
  return current + (target - current) * factor;
}

// Draw face bounding box
function drawFaceBox(face) {
  const start = face.topLeft;
  const end = face.bottomRight;
  const size = [end[0] - start[0], end[1] - start[1]];
  
  ctx.strokeStyle = '#6366F1';
  ctx.lineWidth = 2;
  ctx.strokeRect(start[0], start[1], size[0], size[1]);
  
  // Draw landmarks
  if (face.landmarks) {
    ctx.fillStyle = '#6366F1';
    face.landmarks.forEach(landmark => {
      ctx.beginPath();
      ctx.arc(landmark[0], landmark[1], 2, 0, 2 * Math.PI);
      ctx.fill();
    });
  }
}

// Update emotion history for graphing
function updateEmotionHistory() {
  emotionHistory.stress.push(emotions.stress);
  emotionHistory.excitement.push(emotions.excitement);
  emotionHistory.focus.push(emotions.focus);
  emotionHistory.positive.push(emotions.positive);
  
  // Keep only last 60 seconds
  if (emotionHistory.stress.length > MAX_HISTORY) {
    emotionHistory.stress.shift();
    emotionHistory.excitement.shift();
    emotionHistory.focus.shift();
    emotionHistory.positive.shift();
  }
}

// Get current emotions
function getEmotions() {
  return { ...emotions };
}

// Get emotion history
function getEmotionHistory() {
  return { ...emotionHistory };
}

// Start tracking
async function startTracking() {
  if (isTracking) return;
  
  console.log('=== STARTING ZENTRADER ===');
  
  const cameraReady = await setupCamera();
  if (!cameraReady) {
    console.error('Failed to setup camera');
    return false;
  }
  console.log('Camera ready ✓');
  
  const modelLoaded = await loadModel();
  if (!modelLoaded) {
    console.error('Failed to load model');
    return false;
  }
  console.log('Model loaded ✓');
  
  isTracking = true;
  sessionStartTime = Date.now();
  console.log('Starting detection loop...');
  detectFace();
  
  console.log('=== ZENTRADER STARTED SUCCESSFULLY ===');
  return true;
}

// Stop tracking
function stopTracking() {
  isTracking = false;
  
  if (videoElement && videoElement.srcObject) {
    const tracks = videoElement.srcObject.getTracks();
    tracks.forEach(track => track.stop());
  }
  
  console.log('ZENTRADER stopped');
}

// Export functions
window.EmotionTracker = {
  start: startTracking,
  stop: stopTracking,
  getEmotions,
  getEmotionHistory,
  isTracking: () => isTracking
};
